#ifndef __MGFUNCS
#define __MGFUNCS

#define HOST static __host__ inline
#define DEVICE static __device__ inline

#define GPU
#define SM13

#ifndef GPU
	#define HOSTDEVICE static inline
	#define DBFL double
	#define EPSI 1e-12
	#define PI 3.14159265358979324
	#include <math.h>
#else
	#define HODE_NOINLINE static __device__ __host__ __noinline__
	#define HODE_INLINE static __device__ __host__
#ifndef SM13
	#define DBFL float
	#define EPSI 1e-6
	#define PI 3.141593
#else
	#define DBFL double
	#define EPSI 1e-10
	//#define EPSI 1e-14
	#define PI 3.141592653589793238462643383279502884197169399375105820974944592
#endif
#endif


HODE_NOINLINE DBFL ND2(DBFL a, DBFL b, DBFL rho);
HODE_NOINLINE DBFL tvtl(int jj,DBFL limit[],DBFL sigmarho[],DBFL epsi);
HODE_NOINLINE DBFL derivn3(DBFL limit[],DBFL sigmarho[], int idx);
HODE_NOINLINE DBFL BVTL(int NU, DBFL DH, DBFL DK, DBFL R);
HODE_NOINLINE DBFL TVTMFN(DBFL X, DBFL H1, DBFL H2, DBFL H3, DBFL R23, DBFL RUA, DBFL RUB, DBFL AR, DBFL RUC, int NUC);
HODE_NOINLINE DBFL ADONET(DBFL ZRO,DBFL ONE,DBFL EPS, DBFL H1, DBFL H2,  DBFL H3, DBFL R23, DBFL RUA, DBFL RUB, DBFL AR, DBFL RUC, DBFL NUC);
HODE_NOINLINE void SINCS(DBFL X, DBFL* SX, DBFL* CS );
HODE_NOINLINE DBFL KRNRDT(DBFL A, DBFL B, DBFL* ERR, DBFL H1, DBFL H2,  DBFL H3, DBFL R23, DBFL RUA, DBFL RUB, DBFL AR, DBFL RUC, DBFL NUC );
HODE_NOINLINE DBFL PNTGND(int NUC, DBFL BA, DBFL BB, DBFL BC, DBFL RA, DBFL RB, DBFL R, DBFL RR);


HODE_INLINE DBFL SIGN(DBFL a, DBFL b)
{
	return (b >= 0) ? fabs(a) : -fabs(a);
}

HODE_INLINE DBFL ABS(DBFL x)
{
	return (x < 0) ? -x : x;
}

HODE_INLINE DBFL POW(DBFL x, DBFL y)
{
	return pow(x,y);
}




HODE_INLINE DBFL integr(DBFL t1,DBFL t2)
{
// Function: integr
/*
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!                                                                      !!
!!  Function that evaluates the integral of the risk free interest rate !!
!!  r(t)  on the interval (t1,t2) (user supplied function)              !!
!!                                                                      !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*/
// example risk free interest rate r = 3/100 (constant case)
// the istantaneous rate must be assigned by the user
    return 3.0*(t2-t1)/100.f;
}



HODE_INLINE DBFL integalpha(DBFL t1,DBFL t2)
{
// Function: integalpha
/*
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!                                                                   !!
!!  Function that evaluates the integral of alpha(t) on the interval !!
!!  (t1,t2) (user supplied function)                                 !!
!!                                                                   !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*/
    return 0.01*(t2-t1);
}



HODE_NOINLINE DBFL integs(DBFL t1,DBFL t2)
{
// Function: integs
/*
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!                                                                        !!
!!  Function that evaluates the integral of the square of the  volatility !!
!!  sigma^2(t) on the interval (t1,t2)  (user supplied function)          !!
!!                                                                        !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*/
	DBFL v1 = 0.105, v2 = 0.11478240;

	if (t2 < 0.25)
	{
		return v1*v1*(t2-t1);
	}
	else
	{
		if (t1 >= 0.25) return v2*v2*(t2-t1);
		else return v1*v1*(0.25-t1)+v2*v2*(t2-0.25);
	}
}



HODE_INLINE DBFL sigmaq(DBFL t1)
{
// Function: sigmaq
/*
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!                                                        !!
!!  Function that evaluates the square of the volatility  !!
!!  sigma^2(t) at t=t1                                    !!
!!  (user supplied function)                              !!
!!                                                        !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*/
	 DBFL v1 = 0.105, v2 = 0.11478240;

    if (t1 < 0.25) return v1*v1;
    else return v2*v2;
}



HODE_INLINE DBFL ndf(DBFL t)
{
// standard normal density function
	return 0.398942280401433*exp(-t*t/2);
}

HODE_NOINLINE DBFL nc(DBFL x)
{
// standard normal cumulative distribution function
   DBFL result;
   if (x < -7.0)
      result = ndf(x)/sqrt(1.0+x*x);
   else if (x > 7.0)
      result = 1.0-nc(-x);
   else
   {
      result = 0.2316419;
      DBFL a[5] = {0.31938153,-0.356563782,1.781477937,-1.821255978,1.330274429};
      result = 1.0/(1+result*fabs(x));
      result = 1.0-ndf(x)*(result*(a[0]+result*(a[1]+result*(a[2]+result*(a[3]+result*a[4])))));
      if (x <= 0.0) result = 1.0-result;
   }
   return result;
}


HODE_NOINLINE DBFL PHID(DBFL Z)
{
/*
*     Normal distribution probabilities accurate to 1D-15.
*     Z = number of standard deviations from the mean.
*
*     The software that computes the normal distribution probabilities
*     has been developed by M.C. Recchioni based upon algorithm 5666 (Programmer Alan Miller)
*     for the error function, taken from:
*     Hart, J.F. et al, 'Computer Approximations', Wiley, 1968
*
*/
	DBFL P0, P1, P2, P3, P4, P5, P6;
	DBFL Q0, Q1, Q2, Q3, Q4, Q5, Q6, Q7;
	DBFL P, EXPNTL, CUTOFF, ROOTPI, ZABS;

	P0 = 220.2068679123761;
	P1 = 221.2135961699311;
	P2 = 112.0792914978709;
	P3 = 33.91286607838300;
	P4 = 6.373962203531650;
	P5 = 0.7003830644436881;
	P6 = 0.03526249659989109;

	Q0 = 440.4137358247522;
	Q1 = 793.8265125199484;
	Q2 = 637.3336333788311;
	Q3 = 296.5642487796737;
	Q4 = 86.78073220294608;
	Q5 = 16.064177579206950;
	Q6 = 1.7556671631826420;
	Q7 = 0.088388347648318440;
	ROOTPI = 2.506628274631001;
	CUTOFF = 7.071067811865475;

	ZABS = fabs(Z);
	if (ZABS > 37) P = 0;
	else
	{
		EXPNTL = exp(-ZABS*ZABS/2);
		if (ZABS < CUTOFF)
			P = EXPNTL*((((((P6*ZABS + P5)*ZABS + P4)*ZABS + P3)*ZABS+ P2)*ZABS + P1)*ZABS + P0)/
			(((((((Q7*ZABS + Q6)*ZABS + Q5)*ZABS + Q4)*ZABS + Q3)*ZABS + Q2)*ZABS + Q1)*ZABS + Q0);
		else
			P = EXPNTL/(ZABS + 1./(ZABS + 2./(ZABS + 3./(ZABS + 4./(ZABS + 0.65)))))/ROOTPI;
	}
	if (Z > 0) P = 1 - P;
	return(P);
}



HODE_NOINLINE DBFL fxy(DBFL x, DBFL y, DBFL a, DBFL b, DBFL rho) 
{
//function needed to calculate the two dimensional cumulative distribution function
    DBFL a_s;
    DBFL b_s;
    DBFL result;
    a_s = a / sqrt(2 * (1 - rho * rho));
    b_s = b / sqrt(2 * (1 - rho * rho));
    result = exp(a_s * (2 * x - a_s) + b_s * (2 * y - b_s) + 2 * rho * (x - a_s) * (y - b_s));
    return result;
}



HODE_INLINE DBFL sgn(DBFL a)
{
	// determine the sign of a real number
	if (a > 0) return 1.0 ;
	else if (a < 0) return -1.0;
	else return 0.0;
}



/*
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!                                                              !!
!! Functions needed to compute the  first order term  P_1       !!
!!                                                              !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*/

HODE_NOINLINE DBFL ff(DBFL p,DBFL tt,DBFL a, DBFL b, DBFL gm)
{
/*
!!
!! Function F(p,tt,a,b,gm)
!!
*/
	DBFL aa = -(b*p-b*tt+a)/sqrt(2.0*(tt-p));
	DBFL caux = 2.0*sqrt(PI)*PHID(aa);
	aa = (b*b-(1.0-gm)*(1.0-gm))/4.0;
	return exp(-0.5*a*b)*exp(aa*(tt-p))*caux;
 }

HODE_NOINLINE DBFL v(DBFL p, DBFL tt,DBFL a,DBFL b,DBFL gm)
{
/*
!!
!! Function  E(p,tt,a,b,gm)
!!
*/
	DBFL aa = -(p*(a-b)+b*tt)/sqrt(2.0*p*tt*(tt-p));
	DBFL caux = PHID(aa);

	aa = exp(POW((a-b),2)/(4.0*tt))*exp(POW((1.0-gm),2)*tt/4.0)*sqrt(tt);
	return caux/aa;
}

HODE_NOINLINE DBFL llold(DBFL p,DBFL tt, DBFL a, DBFL b,DBFL c, DBFL gm)
{
/*
!!
!! Fuction L(p,tt,a,b,c,gm)
!!
*/
	DBFL xx = (-a+b*(tt-p))/sqrt(2.0*(tt-p));
	DBFL yy = (-a+b*tt+c)/sqrt(2.0*tt);
	DBFL rho = sqrt((tt-p)/tt);
	DBFL aa = (b*b-(1.0-gm)*(1.0-gm))/4.0;
	DBFL caux = ND2(-xx,-yy,rho);

	return 2.0*sqrt(PI)*exp(-a*b*0.5)*exp(aa*(tt-p))*caux;
}

/*
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!                                                              !!
!! Functions needed to compute the  second order term  P_2      !!
!!                                                              !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*/

HODE_NOINLINE DBFL dvv(DBFL s, DBFL p, DBFL tt, DBFL a, DBFL b, DBFL gm)
{
/*
!!
!! Function  D_E(s,p,tt,a,b,gm)
!!
*/
	DBFL aa,caux,caux1,caux2;
	DBFL xx,yy,rho;

	DBFL p1 = sqrt(2.0*p*tt*(tt-p));
	DBFL p2 = sqrt(2.0*p*tt*(tt-s));
	DBFL p3 = sqrt((s*(tt-p))/(p*(tt-s)));

	aa = (a*p+b*(tt-p))/p1;
	caux = PHID(aa);
	aa = exp((a-b)*(a-b)/(4.0*tt))*exp(POW((1.0-gm),2)*tt/4.0)*sqrt(tt);
	caux = -caux/aa;

	xx = (a*p+b*(tt-p))/p1;
	yy = (a*s+b*(tt-s))/p2;
	rho = p3;
	caux1 = ND2(-xx,-yy,rho);
	caux1 /= aa;

	aa = exp((a+b)*(a+b)/((DBFL)4.0*tt))*exp(POW((1.0-gm),2)*tt/4.0)*sqrt(tt);
	xx=(a*p-b*(tt-p))/p1;
	yy=(a*s-b*(tt-s))/p2;
	rho=p3;
	caux2 = ND2(-xx,-yy,rho);
	caux2 /= aa;

	return (caux+caux1+caux2)/(2.0*sqrt(PI));
}

HODE_NOINLINE DBFL dff(DBFL s, DBFL p, DBFL tt, DBFL a, DBFL b, DBFL gm)
{
/*
!!
!! Function D_F(s,p,tt,a,b,gm)
!!
*/
   DBFL aa,caux,caux1,caux2;
   DBFL xx,yy,rho;

	DBFL p1 = sqrt(2.0*(tt-p));
	DBFL p2 = sqrt(2.0*(tt-s));
	DBFL p3 = sqrt((tt-p)/(tt-s));

   xx = (a-b*(tt-p))/p1;
   caux = -PHID(xx)*exp(-0.5*a*b);

   xx = (a+b*(tt-p))/p1;
   yy = (a+b*(tt-s))/p2;
   rho = p3;
   caux1 = ND2(-xx,-yy,rho);
   caux1 *= exp(0.5*a*b);

   xx = (a-b*(tt-p))/p1;
   yy = (a-b*(tt-s))/p2;
   rho = p3;
   caux2 = ND2(-xx,-yy,rho);
   caux2 *= exp(-0.5*a*b);

   aa = exp((b*b-(1.0-gm)*(1.0-gm))*(tt-s)/4.0);
   return (caux+caux1+caux2)*aa;
}


HODE_NOINLINE DBFL dll(DBFL s, DBFL p, DBFL tt, DBFL a, DBFL b, DBFL c, DBFL gm)
{
/*
!!
!! Function D_L(s,p,a,b,c,gm)
!!
*/
   DBFL aa,caux,caux1;
   DBFL sigmarho[3],limit[3];   

	DBFL p1 = sqrt(2.0*(tt-p));
	DBFL p2 = sqrt(2.0*(tt-s));
	DBFL p3 = sqrt(2.0*tt);
	DBFL rh1 = sqrt((tt-p)/(tt-s));
	DBFL rh2 = sqrt((tt-p)/tt);
	DBFL rh3 = sqrt((tt-s)/tt);

   limit[0] = (a+b*(tt-p))/p1;
   limit[1] = (a+b*(tt-s))/p2;
   limit[2] = (a+b*tt+c)/p3;
   sigmarho[0] = rh1;
   sigmarho[1] = rh2;
   sigmarho[2] = rh3;
   caux = exp(0.5*a*b);
	caux *= tvtl(0,limit,sigmarho,EPSI);

   limit[0] = (a-b*(tt-p))/p1;
   limit[1] = (-a+b*(tt-s))/p2;
   limit[2] = (-a+b*tt+c)/p3;
   sigmarho[0] = -rh1;
   sigmarho[1] = -rh2;
   sigmarho[2] = rh3;
   caux1 = -exp(-0.5*a*b);
	caux1 *= tvtl(0,limit,sigmarho,EPSI);

   aa = exp((b*b-(1.0-gm)*(1.0-gm))*(tt-s)/4.0);
   return (caux+caux1)*aa;
}

HODE_NOINLINE DBFL ddff(DBFL s, DBFL p, DBFL tt, DBFL a, DBFL b, DBFL gm)
{
/*
!!
!! Derivative with respect to a of the function D_F(s,p,tt,a,b,gm)
!!
*/
   DBFL aa,caux,caux1,caux2,caux3,caux4;
   DBFL xx,yy,rho;

	DBFL p1 = sqrt(2.0*(tt-p));
	DBFL p2 = sqrt(2.0*(tt-s));
	DBFL p3 = sqrt(2.0*tt);
	DBFL p4 = (2.0*sqrt(PI*(tt-p)));
	DBFL p5 = (2.0*sqrt(PI*(tt-s)));
	DBFL yy1 = a*sqrt((p-s))/sqrt(2.0*(tt-p)*(tt-s));

	xx=(a-b*(tt-p))/p1;
   caux=PHID(xx)*exp(-0.5*a*b);

   xx=(a+b*(tt-p))/p1;
   yy=(a+b*(tt-s))/p2;
   rho=sqrt((tt-p)/(tt-s));
   caux1=ND2(-xx,-yy,rho);
   caux1=exp(0.5*a*b)*caux1;

   xx=(a-b*(tt-p))/p1;
   yy=(a-b*(tt-s))/p2;
   rho=sqrt((tt-p)/(tt-s));
   caux2=ND2(-xx,-yy,rho);
   caux2=-exp(-0.5*a*b)*caux2;

   caux=0.5*b*(caux+caux1+caux2);


	xx=(a+b*(tt-p))/p1;
   yy=b*sqrt(p-s)/sqrt(2.0);
   caux1=exp(-0.5*xx*xx)*exp(0.5*a*b)*PHID(yy)/p4;

   xx=(a+b*(tt-s))/p2;
   yy=yy1;
   caux2=exp(-0.5*xx*xx)*exp(0.5*a*b)*PHID(yy)/p5;

   xx=(a-b*(tt-p))/p1;
   yy=b*POW((p-s),0.5)/POW(2.0,0.5);
   caux3=-exp(-0.5*xx*xx)*exp(-0.5*a*b)*PHID(yy)/p4;

   xx=(a-b*(tt-s))/p2;
   yy=yy1;
   caux4=exp(-0.5*xx*xx)*exp(-0.5*a*b)*PHID(yy)/p5;


   aa=exp((b*b-(1.0-gm)*(1.0-gm))*(tt-p)/4.0);
   return (caux+caux1+caux2+caux3+caux4)*aa;
}

HODE_NOINLINE DBFL ddll(DBFL s,DBFL p,DBFL tt, DBFL ax, DBFL bx,DBFL c, DBFL gm)
{
/*
!!
!! Derivative with respect to a of the function D_L(s,p,tt,a,b,c,gm)
!!
*/
	DBFL caux,caux1;
	DBFL sigmarho[3],limit[3];

	DBFL l1 = sqrt(2.0*(tt-p));
	DBFL l2 = sqrt(2.0*(tt-s));
	DBFL l3 = sqrt(2.0*tt);
	DBFL r1 = sqrt((tt-p)/(tt-s));
	DBFL r2 = sqrt((tt-p)/tt);
	DBFL r3 = sqrt((tt-s)/tt);

	limit[0]=(ax+bx*(tt-p))/l1;
	limit[1]=(ax+bx*(tt-s))/l2;
	limit[2]=(ax+bx*tt+c)/l3;
	sigmarho[0]=r1;
	sigmarho[1]=r2;
	sigmarho[2]=r3;
	caux=0.5*bx*tvtl(0,limit,sigmarho,EPSI);

   caux += derivn3(limit,sigmarho,1)/l1;
   caux += derivn3(limit,sigmarho,2)/l2;
   caux += derivn3(limit,sigmarho,3)/l3;
	caux *= exp(0.5*ax*bx);

	limit[0]=(ax-bx*(tt-p))/l1;
	limit[1]=(-ax+bx*(tt-s))/l2;
	limit[2]=(-ax+bx*tt+c)/l3;
	sigmarho[0]=-r1;
	sigmarho[1]=-r2;
	sigmarho[2]=r3;
	caux1=0.5*bx*tvtl(0,limit,sigmarho,EPSI);

	caux1 -= derivn3(limit,sigmarho,1)/l1;
	caux1 += derivn3(limit,sigmarho,2)/l2;
	caux1 += derivn3(limit,sigmarho,3)/l3;
	caux1 *= exp(-0.5*ax*bx);

	DBFL aa = exp((bx*bx-(1.0-gm)*(1.0-gm))*(tt-s)/4.0);
	return (caux+caux1)*aa;
}

HODE_NOINLINE DBFL ddvv(DBFL s, DBFL p, DBFL tt, DBFL a, DBFL b, DBFL gm)
{
/*
!!
!!   Derivative with respect to a of the function D_E(s,p,tt,a,b,gm)
!!
*/
   DBFL aa,caux,caux1,caux2,caux6;
   DBFL caux3,caux4,caux5,aux;
   DBFL xx,yy,rho;

	DBFL p1 = sqrt(2.0*tt*p*(tt-p));
	DBFL p2 = sqrt(2.0*tt*s*(tt-s));
	DBFL p3 = sqrt((s*(tt-p))/(p*(tt-s)));
	DBFL d1 = (4.0*p*tt*(tt-p));
	DBFL d2 = (4.0*s*tt*(tt-s));

   aa = (a*p+b*(tt-p))/sqrt(2.0*p*tt*(tt-p));
   caux = PHID(aa);
   aa = exp(-(a-b)*(a-b)/(4.0*tt))/tt;
   caux *= 0.5*aa*(a-b);

   xx = (a*p+b*(tt-p))/p1;
   yy = (a*s+b*(tt-s))/p2;
   rho = p3;
   caux1 = ND2(-xx,-yy,rho);
   caux1 *= -0.5*aa*(a-b);

   aa = exp(-(a+b)*(a+b)/(4.0*tt))/tt;
   xx = (a*p-b*(tt-p))/p1;
   yy = (a*s-b*(tt-s))/p2;
   rho = p3;
   caux2 = ND2(-xx,-yy,rho);
   caux2 *= -0.5*aa*(a+b);

   aa = -b*sqrt((p-s)/sqrt(2.0*p*s));
   aux = sqrt(p/(PI*tt*(tt-p)))*PHID(aa);

	xx = (a+b)*(a+b)/(4.0*tt);
   yy = POW((a*p-b*(tt-p)),2)/d1;
   caux3 = aux*exp(-xx)*exp(-yy)/2.0;

   xx = (a-b)*(a-b)/(4.0*tt);
   yy = POW((a*p+b*(tt-p)),2)/d1;
   caux4 = aux*exp(-xx)*exp(-yy)/2.0;

   aa = a*sqrt((p-s)/sqrt(2.0*(tt-p)*(tt-s)));
   aux = sqrt(s/(PI*tt*(tt-s)))*PHID(aa);

   xx = (a+b)*(a+b)/(4.0*tt);
   yy = POW((a*s-b*(tt-s)),2)/d2;
   caux5 = aux*exp(-xx)*exp(-yy)/2.0;

   xx = (a-b)*(a-b)/(4.0*tt);
   yy = POW((a*s+b*(tt-s)),2)/d2;
   caux6 = aux*exp(-xx)*exp(-yy)/2.0;

   aux = exp((1.0-gm)*(1.0-gm)*tt/4.0)*sqrt(tt);

   return (caux+caux1+caux2+caux3+caux4+caux5+caux6)/(aux*2.0*sqrt(PI));
 }

HODE_NOINLINE DBFL derivn3(DBFL limit[],DBFL sigmarho[], int idx)
{
/*
!!
!! Derivn3 computes the derivatives of the trivariate cumulative normal
!! distribution with respect to one of the integration limits
!!
*/
	DBFL aa;
	DBFL xx,yy,rho,sc;
	DBFL deriv;
	sc=sqrt(2.0*PI);
	switch(idx)
	{
	case 1:
		aa=exp(-0.5*limit[0]*limit[0]);
		xx=(limit[2]-sigmarho[1]*limit[0])/sqrt((1.0-POW(sigmarho[1],2)));
		yy=(limit[1]-sigmarho[0]*limit[0])/sqrt((1.0-POW(sigmarho[0],2)));
		rho=(sigmarho[2]-sigmarho[0]*sigmarho[1])/
			sqrt((1.0-sigmarho[0]*sigmarho[0])*(1.0-sigmarho[1]*sigmarho[1]));
		break;
	case 2:
		aa=exp(-0.5*limit[1]*limit[1]);
		xx=(limit[0]-sigmarho[0]*limit[1])/sqrt((1.0-POW(sigmarho[0],2)));
		yy=(limit[2]-sigmarho[2]*limit[1])/sqrt((1.0-POW(sigmarho[2],2)));
		rho=(sigmarho[1]-sigmarho[0]*sigmarho[2])/
			sqrt((1.0-sigmarho[0]*sigmarho[0])*(1.0-sigmarho[2]*sigmarho[2]));
		break;
	default:
		aa=exp(-0.5*limit[2]*limit[2]);
		xx=(limit[0]-sigmarho[1]*limit[2])/sqrt((1.0-POW(sigmarho[1],2)));
		yy=(limit[1]-sigmarho[2]*limit[2])/sqrt((1.0-POW(sigmarho[2],2)));
		rho=(sigmarho[0]-sigmarho[1]*sigmarho[2])/
			sqrt((1.0-sigmarho[1]*sigmarho[1])*(1.0-sigmarho[2]*sigmarho[2]));
		break;
	}
	deriv=aa*ND2(-xx,-yy,rho)/sc;
	return(deriv);
}



HODE_NOINLINE DBFL tvtl(int NU, DBFL limit[], DBFL sigmarho[],DBFL epsi)
{
/*
      A function for computing trivariate normal and t-probabilities.
      This function uses algorithms developed from the ideas
      described in the papers:
      R.L. Plackett, Biometrika 41(1954), pp. 351-360.
      Z. Drezner, Math. Comp. 62(1994), pp. 289-294.
      and uses adaptive integration.
      The software given here is based on algorithms described in the paper
      A. Genz: "Numerical Computation of Rectangular Bivariate and Trivariate
      Normal and t Probabilities", Statistics and Computing 14 (2004) 251-260.
      This software has been developed by M.C. Recchioni based on previous
      software developed by
      Alan Genz
      Department of Mathematics
      Washington State University
      Pullman, WA 99164-3113
      Email : alangenz@wsu.edu
      The software developed by A. Genz is available free of charge in the website:
      www.math.wsu.edu/faculty/genz/software/software.html

      The software calculates the probability that X(I) < H(I), for I = 1,2,3
      NU        INTEGER degrees of freedom; use NU = 0 for normal cases.
      LIMIT     REAL array of uppoer limits for probability distribution
      SIGMARHO  REAL array of three correlation coefficients, sIGMARHO should
              contain the lower left portion of the correlation matrix.
              SIGMARHO should contains the values r21, r31, r23 in that order.
      EPSI      REAL required absolute accuracy; maximum accuracy for most
              computations is approximately 1D-14

*/

	DBFL ONE=1.0, ZRO=0.0, EPS,  TVT;
	DBFL PT, R12, R23, R13;
	EPS = max(1.e-14, epsi);
	PT = PI/2.0;

	DBFL NUC = NU;
	DBFL H1 = limit[0];
	DBFL H2 = limit[1];
	DBFL H3 = limit[2];
	R12 = sigmarho[0];
	R13 = sigmarho[1];
	R23 = sigmarho[2];

/*
*     Sort R's and check for special cases
*/
	if ( fabs(R12) > fabs(R13) ) 
	{
		H2 = H3;
		H3 = limit[1];
		R12 = R13;
		R13 = sigmarho[0];
	}

	if ( fabs(R13) > fabs(R23) ) 
	{
		H1 = H2;
		H2 = limit[0];
		R23 = R13;
		R13 = sigmarho[2];
	}

	TVT = 0.0;
	if ( (fabs(H1) + fabs(H2) + fabs(H3)) < EPS ) TVT = ( 1 + ( asin(R12) + asin(R13) + asin(R23) )/PT )/8.0;
	else if ( (NU < 1) && ( (fabs(R12) + fabs(R13)) < EPS) )  TVT = PHID(H1)*BVTL( NU, H2, H3, R23 );
	else if ( (NU < 1) && ((fabs(R13) + fabs(R23))< EPS) ) TVT = PHID(H3)*BVTL( NU, H1, H2, R12 );
	else if( (NU < 1) && ((fabs(R12) + fabs(R23))< EPS) ) TVT = PHID(H2)*BVTL( NU, H1, H3, R13 );
	else if ( (1.0 - R23)< EPS ) TVT = BVTL( NU, H1, min( H2, H3 ), R12 );
	else if ( (R23 + 1.0) <EPS ) 
	{
		if  ( H2 > -H3 ) TVT = BVTL( NU, H1, H2, R12 ) - BVTL( NU, H1, -H3, R12 );
	} 
	else 
	{
/*
*        Compute singular TVT value
*/
		if ( NU < 1 ) 
		{
			TVT = BVTL(NU, H2, H3, R23);
			TVT *= PHID(H1);
		}
		else if ( R23 > 0 ) TVT = BVTL( NU, H1, min( H2, H3 ), ZRO );
		else if ( H2 > -H3 ) TVT = BVTL( NU, H1, H2, ZRO ) - BVTL( NU, H1, -H3, ZRO );

/*
*        Use numerical integration to compute probability
*
*/
		DBFL RUA = asin( R12 );
		DBFL RUB = asin( R13 );
		DBFL AR = asin( R23);
		DBFL RUC = SIGN( PT, AR ) - AR;
		TVT = TVT + ADONET(  ZRO, ONE, EPS, H1, H2, H3, R23, RUA, RUB, AR, RUC, NUC )/( 4.0*PT );
	}
	return max( ZRO, min( TVT, ONE ) );
}


HODE_NOINLINE DBFL TVTMFN(DBFL X, DBFL H1, DBFL H2, DBFL H3, DBFL R23, DBFL RUA, DBFL RUB, DBFL AR,DBFL RUC, int NUC )
{
/*
       Computes Plackett formula integrands
*/

      DBFL R12=0.0, RR2=0, R13=0.0, RR3=0.0, R=0.0, RR=0.0, ZRO=0.0;
      DBFL result = 0.0;

      SINCS( RUA*X, &R12, &RR2 );
      SINCS( RUB*X, &R13, &RR3 );

      if ( fabs(RUA)> 0 ) result += RUA*PNTGND( NUC, H1,H2,H3, R13,R23,R12,RR2 );
      if ( fabs(RUB)> 0 ) result += RUB*PNTGND( NUC, H1,H3,H2, R12,R23,R13,RR3 );
      if ( NUC > 0 )
      {
         SINCS( AR + RUC*X, &R, &RR );
         result -= RUC*PNTGND( NUC, H2, H3, H1, ZRO, ZRO, R, RR );
      }
      return(result);
}

HODE_NOINLINE void SINCS(DBFL X, DBFL* SX, DBFL* CS)
{
/*
     Computes SIN(X), COS(X)^2, with series approx. for |X| near PI/2
*/
	//PT = 1.57079632679489661923132169163975;
	DBFL PT = PI / 2;
	DBFL EE = POW(( PT - fabs(X) ),2);

	if ( EE < 5e-5 )
	{
		*SX = SIGN( 1 - EE*( 1 - EE/12 )/2, X );
		*CS = EE*( 1 - EE*( 1 - 2*EE/15 )/3 );
	}
	else
	{
		*SX = sin(X);
		*CS = 1 - sin(X)*sin(X);
	}
	return;
	}

HODE_NOINLINE DBFL ADONET(DBFL A,DBFL B, DBFL TOL, DBFL H1, DBFL H2,  DBFL H3, DBFL R23, DBFL RUA, DBFL RUB, DBFL AR, DBFL RUC, DBFL NUC)
{
	// One Dimensional Globally Adaptive Integration Function

	const int NL = 100; 
	int I, IM, IP;
	DBFL EI[NL+1], AI[NL+1], BI[NL+1], FI[NL+1], FIN;
	DBFL ERR;

	for (int i=0; i<=NL; i++) 
	{
		EI[i] = 0;
		AI[i] = 0;
		BI[i] = 0;
		FI[i] = 0;
	}

	AI[1] = A;
	BI[1] = B;
	ERR = 1;
	IP = 1;
	IM = 1;
	while (((4*ERR) > TOL) && (IM < NL))
	{
		IM = IM + 1;
		BI[IM] = BI[IP];
		AI[IM] = (AI[IP] + BI[IP] )/2.0;
		BI[IP] = AI[IM];
		FI[IP] = KRNRDT( AI[IP], BI[IP], &EI[IP], H1, H2, H3, R23, RUA, RUB, AR, RUC, NUC );
		FI[IM] = KRNRDT( AI[IM], BI[IM], &EI[IM], H1, H2, H3, R23, RUA, RUB, AR, RUC, NUC );

		ERR = 0.0;
		FIN = 0.0;
		for(I=1; I<=IM; I++)
		{
			if( EI[I] > EI[IP]) IP = I;
			FIN = FIN + FI[I];
			ERR = ERR + EI[I]*EI[I];
		}
		ERR = POW( ERR,0.5 );
	}
	return FIN;
}

HODE_NOINLINE DBFL KRNRDT(DBFL A, DBFL B, DBFL* ERR, DBFL H1, DBFL H2,  DBFL H3, DBFL R23, DBFL RUA, DBFL RUB, DBFL AR, DBFL RUC, DBFL NUC )
{
//
//     Kronrod Rule
//
      DBFL T, CEN, FC, WID, RESG, RESK;
      DBFL result;
//
//        The abscissae and weights are given for the interval (-1,1);
//        only positive abscissae and corresponding weights are given.
//
//        XGK    - abscissae of the 2N+1-point Kronrod rule:
//                 XGK(2), XGK(4), ...  N-point Gauss rule abscissae;
//                 XGK(1), XGK(3), ...  optimally added abscissae.
//        WGK    - weights of the 2N+1-point Kronrod rule.
//        WG     - weights of the N-point Gauss rule.
//
      int J, N=11;
      DBFL WG[7], WGK[12], XGK[12];

      WG[0]= 0.2729250867779007;
      WG[1]= 0.05566856711617449;
      WG[2]= 0.1255803694649048;
      WG[3]= 0.1862902109277352;
      WG[4]= 0.2331937645919914;
      WG[5]= 0.2628045445102478;
		WG[6]= 0;
//
      XGK[0]= 0.0000000000000000;
      XGK[1]= 0.9963696138895427;
      XGK[2]= 0.9782286581460570;
      XGK[3]= 0.9416771085780681;
      XGK[4]= 0.8870625997680953;
      XGK[5]= 0.8160574566562211;
      XGK[6]= 0.7301520055740492;
      XGK[7]= 0.6305995201619651;
      XGK[8]= 0.5190961292068118;
      XGK[9]= 0.3979441409523776;
      XGK[10]=0.2695431559523450;
      XGK[11]=0.1361130007993617;
//
      WGK[0]= 0.1365777947111183;
      WGK[1]= 0.9765441045961290e-02;
      WGK[2]= 0.2715655468210443e-01;
      WGK[3]= 0.4582937856442671e-01;
      WGK[4]= 0.6309742475037484e-01;
      WGK[5]= 0.7866457193222764e-01;
      WGK[6]= 0.9295309859690074e-01;
      WGK[7]= 0.1058720744813894;
      WGK[8]= 0.1167395024610472;
      WGK[9]= 0.1251587991003195;
      WGK[10]=0.1312806842298057;
      WGK[11]=0.1351935727998845;
/*
	Major variables

	CEN  - mid point of the interval
	WID  - half-length of the interval
	RESG - result of the N-point Gauss formula
	RESK - result of the 2N+1-point Kronrod formula
	Compute the 2N+1-point Kronrod approximation to
	the integral, and estimate the absolute error.
*/
      WID = (B - A) / 2.0;
      CEN = (B + A) / 2.0;

      FC = TVTMFN(CEN, H1, H2, H3, R23, RUA, RUB, AR, RUC, NUC);

      RESG = FC*WG[0];
      RESK = FC*WGK[0];

      for (J=1; J<=N; J++)
      {
         T = WID*XGK[J];
         FC = TVTMFN(CEN-T, H1, H2, H3, R23, RUA, RUB, AR, RUC, NUC) + TVTMFN(CEN+T, H1, H2, H3, R23, RUA, RUB, AR, RUC, NUC);
         RESK = RESK + WGK[J]*FC;
         if((J%2) == 0) RESG += WG[J/2]*FC;
      }
      result = WID*RESK;
      *ERR = fabs( WID*( RESK - RESG ) );
      return(result);
}

HODE_NOINLINE DBFL STUDNT(int NU, DBFL T )
{
/*
     Student t Distribution Function
*/
      int J;
      DBFL  ZRO=0.0, ONE=1.0;
      DBFL  CSSTHE, SNTHE, POLYN, TT, TS, RN;
      DBFL result;


      if ( NU < 1 ) result= PHID( T );
      else if ( NU == 1 ) result = ( 1 + 2.0*atan(T)/PI )/2.0;
      else if ( NU == 2 ) result = ( 1 + T/POW(( 2.0 + T*T),0.5))/2.0;
      else
      {
         TT = T*T;
         CSSTHE = 1/( 1 + TT/DBFL(NU) );
         POLYN = 1;
         for( J = NU-2; J>= 2; J=J-2)
         {
             POLYN = 1.0 + ( J - 1.0 )*CSSTHE*POLYN/(DBFL)J;
	 }
         if ((NU-2*int(NU/2) ) == 1 )
         {
            RN = NU;
            TS = T/POW(RN,0.5);
            result = ( 1.0 + 2.0*( atan(TS) + TS*CSSTHE*POLYN )/PI )/2.0;
         }
         else
         {
            SNTHE = T/POW(( NU + TT ),0.5);
            result = ( 1 + SNTHE*POLYN )/2.0;
	 }
         result = max( ZRO, min( result, ONE ) );
      }
      return(result);
  }

HODE_NOINLINE DBFL BVTL(int NU, DBFL DH, DBFL DK, DBFL R )
{
/*
       A function for computing bivariate t probabilities.
       This function is based on the method described by
       Dunnett, C.W. and M. Sobel, (1954),
       A bivariate generalization of Student's t-distribution
       with tables for certain special cases,
       Biometrika 41, pp. 153-169.
       The software given here has been developed by M.C. Recchioni based on previous
       software developed by
       Alan Genz
       Department of Mathematics
       Washington State University
       Pullman, WA 99164-3113
       Email : alangenz@wsu.edu
       The software developed by A. Genz is available free of charge in
       the website: www.math.wsu.edu/faculty/genz/software/software.html
***

 BVTL - calculate the probability that X < DH and Y < DK.

 parameters

      NU number of degrees of freedom
      DH 1st lower integration limit
      DK 2nd lower integration limit
      R   correlation coefficient
*/
	int  J, HS, KS;
	DBFL  TPI, ORS, HRK, KRH, BVT, SNU;
	DBFL  GMPH, GMPK, XNKH, XNHK, QHRK, HKN, HPK, HKRN;
	DBFL  BTNCKH, BTNCHK, BTPDKH, BTPDHK, ONE;
	DBFL result;
	ONE = 1;

	if ( NU <1 ) 
	{
		result = ND2( -DH, -DK, R );
	}
	else if ( (1 - R)<= EPSI ) result = STUDNT( NU, min( DH, DK ) );
	else	if( (R + 1)<=EPSI )
	{
		if( DH > -DK ) result = STUDNT( NU, DH ) - STUDNT( NU, -DK );
		else result = 0.0;
	}
	else
	{
		TPI = 2.0*PI;
		SNU = (DBFL)NU;
		SNU = POW(SNU,0.5);
		ORS = 1.0 - R*R;
		HRK = DH - R*DK;
		KRH = DK - R*DH;
		if((fabs(HRK) + ORS)> 0 )
		{
			XNHK = HRK*HRK/( HRK*HRK + ORS*( NU + DK*DK ) );
			XNKH = KRH*KRH/( KRH*KRH+ ORS*( NU + DH*DH ) );
		}
		else
		{
			XNHK = 0.0;
			XNKH = 0.0;
		}

		HS =(int)SIGN( ONE, DH - R*DK );
		KS =(int)SIGN( ONE, DK - R*DH );
		if((NU-2*(int)(NU/2))==0 )
		{
			BVT = atan2( POW(ORS,0.5), -R )/TPI;
			GMPH = DH/POW( 16*( NU + DH*DH ),0.5 );
			GMPK = DK/POW( 16*( NU + DK*DK),0.5);
			BTNCKH = 2*atan2( POW( XNKH,0.5 ), POW(( 1-XNKH),0.5) )/PI;
			BTPDKH = 2*POW( XNKH*( 1 - XNKH ),0.5 )/PI;
			BTNCHK = 2*atan2( POW( XNHK,0.5 ), POW((1 - XNHK),0.5) )/PI;
			BTPDHK = 2*POW( XNHK*( 1 - XNHK ),0.5 )/PI;
			for( J = 1; J<= NU/2;J++)
			{
				BVT = BVT + GMPH*( 1 + KS*BTNCKH );
				BVT = BVT + GMPK*( 1 + HS*BTNCHK );
				BTNCKH = BTNCKH + BTPDKH;
				BTPDKH = 2*J*BTPDKH*( 1 - XNKH )/( 2*J + 1 );
				BTNCHK = BTNCHK + BTPDHK;
				BTPDHK = 2*J*BTPDHK*( 1 - XNHK )/( 2*J + 1 );
				GMPH = GMPH*( 2*J - 1 )/( 2*J*( 1 + DH*DH/NU ) );
				GMPK = GMPK*( 2*J - 1 )/( 2*J*( 1 + DK*DK/NU ) );
			}
		}
		else
		{
			QHRK = POW((DH*DH + DK*DK - 2*R*DH*DK + NU*ORS),0.5 ) ;
			HKRN = DH*DK + R*NU ;
			HKN = DH*DK - NU;
			HPK = DH + DK;
			BVT = atan2( -SNU*( HKN*QHRK + HPK*HKRN ),HKN*HKRN-NU*HPK*QHRK )/TPI;
			if ( BVT < -EPSI ) BVT = BVT + 1;
			GMPH = DH/( TPI*SNU*( 1 + DH*DH/NU ) );
			GMPK = DK/( TPI*SNU*( 1 + DK*DK/NU ) );
			BTNCKH = POW( XNKH,0.5 );
			BTPDKH = BTNCKH;
			BTNCHK = POW( XNHK,0.5 );
			BTPDHK = BTNCHK;
			for( J = 1;J<= ( NU - 1 )/2; J++)
			{
				BVT = BVT + GMPH*( 1 + KS*BTNCKH );
				BVT = BVT + GMPK*( 1 + HS*BTNCHK );
				BTPDKH = ( 2*J - 1 )*BTPDKH*( 1 - XNKH )/( 2*J );
				BTNCKH = BTNCKH + BTPDKH;
				BTPDHK = ( 2*J - 1 )*BTPDHK*( 1 - XNHK )/( 2*J );
				BTNCHK = BTNCHK + BTPDHK;
				GMPH = 2*J*GMPH/( ( 2*J + 1 )*( 1 + DH*DH/NU ) );
				GMPK = 2*J*GMPK/( ( 2*J + 1 )*( 1 + DK*DK/NU ) );
			}
		}
		result = BVT;
	}
	
	return(result);
}


HODE_NOINLINE DBFL PNTGND(int NUC, DBFL BA, DBFL BB, DBFL BC, DBFL RA, DBFL RB, DBFL R, DBFL RR)
{
/*
     Computes Plackett formula integrand
*/
      DBFL DT, FT, BT,result;

      result = 0.0;
      DT = RR*( RR - POW(( RA - RB ),2) - 2*RA*RB*( 1 - R ) );
      if( DT > 0 )
      {
         BT = ( BC*RR + BA*( R*RB - RA ) + BB*( R*RA -RB ) )/POW(DT,0.5);
         FT = POW(( BA - R*BB ),2)/RR + BB*BB;
         if( NUC<1 )
	 {
	     if ( (BT > -10) && (FT <100) )
             {
                result = exp( -FT/2 );
                if ( BT <10 ) result= result*PHID(BT);
	     }
             else
	     {
                FT = POW((1 + FT/NUC),0.5);
                result = STUDNT( NUC, BT/FT )/POW(FT,NUC);
	     }
	 }
       }
       return(result);
 }



HODE_NOINLINE DBFL ND2(DBFL a, DBFL b, DBFL rho )
{
/*
*     A function for computing bivariate normal probabilities.
*     This function is based on the method described by
*     Z. Drezner and G.O. Wesolowsky, (1989),
*     On the computation of the bivariate normal integral,
*     Journal of Statist. Comput. Simul. 35, pp. 101-107,
*     with major modifications for DBFL precision, and for |R| close to 1.
*     The software given here has been developed by M.C. Recchioni based on previous
*     software developed by:
*     Alan Genz
*     Department of Mathematics
*     Washington State University
*     Pullman, WA 99164-3113
*     Email : alangenz@wsu.edu
*     The software developed by A. Genz is available free of charge in the website:
*     www.math.wsu.edu/faculty/genz/software/software.html
*
*
*      ND2 calculates the probability that X > DH and Y > DK.
*      Note: Prob( X < DH, Y < DK ) = ND2( -DH, -DK, R ).
*
* Parameters
*
*   DH  DBFL PRECISION, integration limit
*   DK  DBFL PRECISION, integration limit
*   R   DBFL PRECISION, correlation coefficient
*/
	DBFL TWOPI = 2*PI;
	DBFL DK, DH, R;
	int I, IS, LG, NG;
	DBFL XL[10][3], WL[10][3], AS, AA, BB, C, D, RS, XS, BVN;
	DBFL SN, ASR, H, K, BS, HS, HK;

//  Gauss Legendre Points and Weights, N =  6
//  DATA ( W(I,1), X(I,1), I = 1,3) /
	WL[0][0]=0.1713244923791705;
	XL[0][0]=-0.9324695142031522;
	WL[1][0]= 0.3607615730481384;
	XL[1][0]=-0.6612093864662647;
	WL[2][0]= 0.4679139345726904;
	XL[2][0]=-0.2386191860831970;

//  Gauss Legendre Points and Weights, N = 12
//  DATA ( W(I,2), X(I,2), I = 1,6) /
	WL[0][1]=0.4717533638651177e-01;
	XL[0][1]=-0.9815606342467191;
	WL[1][1]=  0.1069393259953183;
	XL[1][1]=-0.9041172563704750;
	WL[2][1]=  0.1600783285433464;
	XL[2][1]=-0.7699026741943050;
	WL[3][1]=  0.2031674267230659;
	XL[3][1]=-0.5873179542866171;
	WL[4][1]=  0.2334925365383547;
	XL[4][1]=-0.3678314989981802;
	WL[5][1]= 0.2491470458134029;
	XL[5][1]=-0.1252334085114692;

//  Gauss Legendre Points and Weights, N = 20
//  DATA ( W(I,3), X(I,3), I = 1, 10 ) /
	WL[0][2]=0.1761400713915212e-01;
	XL[0][2]=-0.9931285991850949;
	WL[1][2]=0.4060142980038694e-01;
	XL[1][2]=-0.9639719272779138;
	WL[2][2]=0.6267204833410906e-01;
	XL[2][2]=-0.9122344282513259;
	WL[3][2]=0.8327674157670475e-01;
	XL[3][2]=-0.8391169718222188;
	WL[4][2]=0.1019301198172404;
	XL[4][2]=-0.7463319064601508;
	WL[5][2]=0.1181945319615184;
	XL[5][2]=-0.6360536807265150;
	WL[6][2]=0.1316886384491766;
	XL[6][2]=-0.5108670019508271;
	WL[7][2]=0.1420961093183821;
	XL[7][2]=-0.3737060887154196;
	WL[8][2]=0.1491729864726037;
	XL[8][2]=-0.2277858511416451;
	WL[9][2]=0.1527533871307259;
	XL[9][2]=-0.7652652113349733e-01;

	R=rho;
	DH=a;
	DK=b;

	if( fabs(R) < 0.3 ) 
	{
		NG = 1;
		LG = 3; 
	}
	else if ( fabs(R) < 0.75 ) 
	{
		NG = 2;
		LG = 6;
	}
	else
	{
		NG = 3;
		LG = 10;
	}
	H = DH;
	K = DK;
	HK = H*K;

	BVN = 0.0 ;
	if( fabs(R) < 0.925 ) 
	{
		if( fabs(R) > 0 ) 
		{
			HS = ( H*H + K*K )/2;
			ASR = asin(R);
			for (I = 1;I<= LG; I++)
			{
				for( IS = -1; IS<= 1; IS=IS+2)
				{
					SN = sin( ASR*(  IS*XL[I-1][NG-1] + 1 )/2 );
					BVN = BVN + WL[I-1][NG-1]*exp( ( SN*HK-HS )/( 1.0-SN*SN ) );
				}
			}
			BVN = BVN*ASR/( 2*TWOPI );
		}
		BVN = BVN + PHID(-H)*PHID(-K);
	}
	else
	{
		if ( R < 0 ) 
		{
			K = -K;
			HK = -HK;
		}

		if( fabs(R) <1 ) 
		{
			AS = ( 1 - R )*( 1 + R );
			AA = POW(AS,0.5);

			BS = POW(( H - K ),2);
			C = ( 4 - HK )/8 ;
			D = ( 12 - HK )/16;
			ASR = -( BS/AS + HK )/2;
			if( ASR > -100 ) BVN = AA*exp(ASR)*( 1 - C*( BS - AS )*( 1 - D*BS/5 )/3 + C*D*AS*AS/5 );
			if( -HK<100 )
			{
				BB = POW(BS,0.5);
				BVN = BVN - exp( -HK/2 )*POW(TWOPI,0.5)*PHID(-BB/AA)*BB*( 1 - C*BS*( 1 - D*BS/5 )/3 );
			}
			AA = AA/2	;
			for (I = 1; I<= LG;I++)
			{
				for( IS = -1; IS<=1; IS=IS+2)
				{
					XS =POW( ( AA*(  IS*XL[I-1][NG-1] + 1 ) ),2);
					RS = POW( (1 - XS),2 );
					ASR = -( BS/XS + HK )/2;
					if ( ASR > -100 ) 
					{
						BVN = BVN + AA*WL[I-1][NG-1]*exp( ASR )*(exp( -HK*( 1 - RS )/( 2*( 1 + RS ) ) )/RS- ( 1 + C*XS*( 1 + D*XS ) ) );
					}
				}
			}
			BVN = -BVN/TWOPI;
		}
		if ( R > 0 )  
		{
			BVN =  BVN + PHID( -max( H, K ) );
		}
		else
		{
			BVN = -BVN;
			if( K > H ) BVN = BVN + PHID(K) - PHID(H);
		}
	}
	
	return BVN;
}

#endif